<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use App\Models\User;

class RoleController extends Controller
{
    // 1. List all Roles
    public function index()
    {
        // Load roles with their associated users
        $roles = Role::with(['permissions', 'users'])->get();

        // Get all users so we can populate the dropdowns
        $users = User::all();

        return view('admin.roles.index', compact('roles', 'users'));
    }

    // 2. Show Create Form
    public function create()
    {
        $permissions = Permission::all();
        return view('admin.roles.create', compact('permissions'));
    }

    // 3. Store New Role
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|unique:roles,name',
            'permissions' => 'array'
        ]);

        $role = Role::create(['name' => $request->name]);

        if ($request->has('permissions')) {
            $role->syncPermissions($request->permissions);
        }

        return redirect()->route('admin.roles.index')->with('success', 'Role created successfully');
    }

    // 4. Show Edit Form
    public function edit($id)
    {
        $role = Role::findOrFail($id);
        $permissions = Permission::all();
        // Get permissions currently assigned to this role
        $rolePermissions = $role->permissions->pluck('id')->toArray();

        return view('admin.roles.edit', compact('role', 'permissions', 'rolePermissions'));
    }

    // 5. Update Role
    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required',
            'permissions' => 'array'
        ]);

        $role = Role::findOrFail($id);
        $role->update(['name' => $request->name]);

        if ($request->has('permissions')) {
            $role->syncPermissions($request->permissions);
        }

        return redirect()->route('admin.roles.index')->with('success', 'Role updated successfully');
    }

    // 6. Delete Role
    public function destroy($id)
    {
        DB::table("roles")->where('id', $id)->delete();
        return redirect()->route('admin.roles.index')->with('success', 'Role deleted successfully');
    }

    public function assignUser(Request $request, Role $role)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
        ]);

        $user = User::findOrFail($request->user_id);

        // Spatie Syntax: Assign the role
        $user->assignRole($role);

        return back()->with('success', "{$user->name} is now a {$role->name}.");
    }

    // NEW: Remove a user from a role
    public function removeUser(Role $role, $userId)
    {
        $user = User::findOrFail($userId);

        // Spatie Syntax: Remove the role
        $user->removeRole($role);

        return back()->with('success', "Removed {$user->name} from {$role->name}.");
    }

}
